#include "MyInfoWidget.h"

#include "MyCamera.h"
#include "MyViewfinderX.h"
#include "glShapes.h"

#include <FCam/Time.h>

MyInfoWidget::MyInfoWidget(MyViewfinderX * parent, ShotMode mode, int numRings, int spotsPerRing) 
: QGLWidget(parent) {

  m_shotMode = mode;

  m_programObject = -1; 

  // TODO(edluong): not sure if this is right
  static const int width = 640;
  static const int height = 480;
  static const int hWidth = width/2;
  static const int hHeight = height/2;

  static const int kMainHotSpotSide = 100;
  static const int kSmallHotSpotSide = 40;

  HotSpotElem main;
  main.box.bmin[0] = hWidth  - kMainHotSpotSide/2;
  main.box.bmax[0] = hWidth  + kMainHotSpotSide/2;

  main.box.bmin[1] = hHeight - kMainHotSpotSide/2;
  main.box.bmax[1] = hHeight + kMainHotSpotSide/2;
  main.visited = false;

  m_hotSpots.push_back(main);

  float k = 0.5f*(kMainHotSpotSide + kSmallHotSpotSide);
  float minRadius = 1.8f*k;
  float maxRadius = 3.f*k;
  float dRadius = (maxRadius - minRadius) / (numRings - 1);

  float dTheta = 2.f*M_PI / spotsPerRing;

  float radius = minRadius;
  for (int ring = 0; ring < numRings; ring++, radius += dRadius) {
    float theta = ring % 2 ? (0.5*dTheta) : 0.f; 
    for (int spot = 0; spot < spotsPerRing; spot++, theta += dTheta) {

      float centerX = hWidth + radius * cos(theta);
      float centerY = hHeight + radius * sin(theta);

      HotSpotElem other;
      other.box.bmin[0] = centerX - kSmallHotSpotSide/2;
      other.box.bmax[0] = centerX + kSmallHotSpotSide/2;
      other.box.bmin[1] = centerY - kSmallHotSpotSide/2;
      other.box.bmax[1] = centerY + kSmallHotSpotSide/2;
      other.visited = false;

      m_hotSpots.push_back(other);
    }
  }
}

void 
MyInfoWidget::viewfinderMouseMove(int x, int y) {
  int which = -1;
  if (m_shotMode == HotSpot && isOverHotSpot(x, y, which)) {
    FCam::Event ev;
    ev.type = which == 0 ? MyCamera::CaptureHighRes : MyCamera::CaptureLoRes;
    ev.time = FCam::Time::now();
    ev.iData = which;
    fprintf(stderr, "ev gen: %d\n", ev.iData);
    postEvent(ev);
    return;
  }
}

void MyInfoWidget::processFrame(FCam::Frame::Ptr f) {	
  m_frame = f;
  /*
   *
   *
   * TODO : Extract whatever information you deem
   * useful or necessary. In the very least,
   * the user should be informed when the focus is locked.
   *
   * To that end, it may be helpful to introduce additional
   * methods, such as
   *
   * void MyInfoWidget::focusLocked(bool b) {
   *    m_focusLocked = b;
   * }
   *
   */ 
  if (m_frame) update();
}

void MyInfoWidget::paintGL() {
  // Allocate OpenGL ES shader object
  if (m_programObject < 0) createProgramObject();
  
  int w = width(), h = height();

  qglClearColor(MyViewfinderX::colorKey());
  glViewport(0, 0, w, h);
  glClear(GL_COLOR_BUFFER_BIT);

  if (!m_frame) return;

  // Prepare for drawing
  glUseProgram(m_programObject);    
  glLineWidth(2);
  glDisable(GL_DITHER);

  Matrix proj, mv;
  Matrix::Ortho2D(proj, 0, w, h, 0);
  //Matrix::Identity(proj);
  glUniformMatrix4fv(m_projMatrixLoc, 1, GL_FALSE, proj.getValue());

  Matrix::Identity(mv);
  glUniformMatrix4fv(m_mvMatrixLoc, 1, GL_FALSE, mv.getValue());

  if (m_shotMode == HotSpot) {
    DrawHotSpots();
  }
}

void
MyInfoWidget::DrawHotSpots() const
{
  for (size_t i = 0; i < m_hotSpots.size(); i++) {
    const BBox2s& box = m_hotSpots[i].box;
    if (m_hotSpots[i].visited) {
      glVertexAttrib3f(COL_ATTR, 1.f, .4f, 0.f);
    }
    else {
      glVertexAttrib3f(COL_ATTR, 1.f, 1.f, 1.f);
    }
    DrawBox(box);
  }
}

bool
MyInfoWidget::isOverHotSpot(int x, int y, int& which)
{
  bool checkBoxes = true;

  if (!checkBoxes) return false;

  Vec2s p(x, y);
  for (size_t i = 0; i < m_hotSpots.size(); i++) {
    if (!m_hotSpots[i].visited && m_hotSpots[i].box.Contains(p)) {
      which = i;
      m_hotSpots[i].visited = true;
      return true;
    }
  }
  return false;
}

void MyInfoWidget::createProgramObject() {
  // OpenGL ES basic shaders boilerplate
  GLuint vertShader = glCreateShader(GL_VERTEX_SHADER);
  const char vertSource[] = 
    "uniform mat4 projection;\n"
    "uniform mat4 modelview;\n"
    "attribute vec4 position, color;\n"
    "varying vec4 fragColor;\n"
    "void main() {\n"
    "   gl_Position = projection * modelview * position;\n"
    "   fragColor = color;\n"
    "}\n";
  const char *sourcePtr = &(vertSource[0]);
  glShaderSource(vertShader, 1, &sourcePtr, NULL);
  glCompileShader(vertShader);
  
  GLuint fragShader = glCreateShader(GL_FRAGMENT_SHADER);
  const char fragSource[] = 
    "precision mediump float;\n"
    "varying vec4 fragColor;\n"
    "void main() {\n"
    "   gl_FragColor = fragColor;\n"
    "}\n";
  
  sourcePtr = &(fragSource[0]);
  glShaderSource(fragShader, 1, &sourcePtr, NULL);
  glCompileShader(fragShader);
  
  GLint compileStatus = 0;
  glGetShaderiv(fragShader, GL_COMPILE_STATUS, &compileStatus);
  if (!compileStatus) {
    fprintf(stderr, "MyInfoWidget : Frag shader didn't compile\n");
    GLint infoLen;
    glGetShaderiv(fragShader, GL_INFO_LOG_LENGTH, &infoLen);		
    char *infoLog = new char[infoLen];
    glGetShaderInfoLog(fragShader, infoLen, NULL, infoLog);
    printf(infoLog);
    delete[] infoLog;		
  }
  glGetShaderiv(vertShader, GL_COMPILE_STATUS, &compileStatus);
  if (!compileStatus) {
    fprintf(stderr, "MyInfoWidget : Vert shader didn't compile\n");
    GLint infoLen;
    glGetShaderiv(vertShader, GL_INFO_LOG_LENGTH, &infoLen);		
    char *infoLog = new char[infoLen];
    glGetShaderInfoLog(vertShader, infoLen, NULL, infoLog);
    printf(infoLog);
    delete[] infoLog;
  }
  
  m_programObject = glCreateProgram();
  glAttachShader(m_programObject, vertShader);
  glAttachShader(m_programObject, fragShader);
  
  glBindAttribLocation(m_programObject, POS_ATTR, "position");
  glBindAttribLocation(m_programObject, COL_ATTR, "color");
  
  glLinkProgram(m_programObject);
  fprintf(stdout, "MyInfoWidget : Vertex and Fragment shaders compiled\n");

  m_projMatrixLoc = glGetUniformLocation(m_programObject, "projection");
  m_mvMatrixLoc = glGetUniformLocation(m_programObject, "modelview");
}

