#include "MyCameraThread.h"

#include "MyCamera.h"

#include <FCam/AsyncFile.h>
#include <FCam/processing/JPEG.h>

static const int JPEG_QUALITY = 90;

MyCameraThread::MyCameraThread(MyViewfinderX * viewfinder,
    MyInfoWidget * infowidget,
    QObject * parent)
: QThread(parent)
{
  m_shotMode = HotSpot;

  m_terminate = false;
  m_viewfinder = viewfinder;
  m_infowidget = infowidget;

  /* We will be requesting a frame from the sensor continuously.
   * The request consists of a set of parameters, encapculated
   * in the class FCam::Shot.
   */
  m_shot.exposure = 20000;
  m_shot.frameTime = 40000;
  m_shot.gain = 1.0f;
  m_shot.image = m_viewfinder->getFrameBuffer(640, 480);
  m_shot.sharpness.enabled = true;
  m_shot.sharpness.size = FCam::Size(16, 12);

  /* We need a separate set of parameters for a full capture. */
  m_snapshot.exposure = 20000;
  m_snapshot.frameTime = 40000;
  m_snapshot.gain = 1.0f;
  m_snapshot.image = FCam::Image(2592, 1968, FCam::UYVY, FCam::Image::AutoAllocate);
}

MyCameraThread::~MyCameraThread() {
  // Nothing to destroy currently.
}

unsigned int
ComputeSharpness(const FCam::Frame::Ptr f)
{
  const FCam::Shot& shot = f->shot();

  unsigned int totalSharpness = 0;
  for (int y = 0; y < shot.sharpness.size.height; y++) {
    for (int x = 0; x < shot.sharpness.size.width; x++) {
      totalSharpness += (f->sharpness(x, y) << 5);
    }
  }
  return totalSharpness;
}

void
MySaveJPEG(FCam::Frame::Ptr f, const char* fname, int quality)
{
  // Should gamma correct.. but I'll just do that later
  saveJPEG(f, fname, quality);
}

void MyCameraThread::run() {
  fprintf(stderr, "MyCameraThread : starting camera thread.\n");
  // These represents the respective hardware.
  FCam::N900::Flash flash;
  FCam::N900::Sensor sensor;
  FCam::AsyncFileWriter writer;
  sensor.attach(&m_lens);
  sensor.attach(&flash);

  focusFar();

  /* Request the sensor to continuously stream frames
   * using the parameters from m_shot. This makes
   * a copy of the shot and sends it to the sensor.
   * Thus, if a parameter changes, you must inform
   * the sensor of this change explicitly.
   */
  sensor.stream(m_shot);

  // This flag controls whether we should take a snapshot or not.

  /* Loop to process the frames */
  int scheduledShot = -1;
  int numStreamed = -1;
  bool takeSnapshot;

  fprintf(stderr, "MyCameraThread : entering streaming loop.\n");
  while (!m_terminate) {

    // If the shot parameter has changed, inform the sensor.
    if (m_exposureUpdated) {
      sensor.stream(m_shot); // Send a stream request again with new parameters.
      m_exposureUpdated = false;
    }

    // FCam library will send us events when they occur. Deal with them.
    FCam::Event e;
    while (getNextEvent(&e)) { // getNextEvent is implemented in Event.h.
      // e.description() is a string description of the event.
      switch(e.type) {
        case MyCamera::CaptureHighRes:
          takeSnapshot = true;
          break;
        case MyCamera::CaptureLoRes:
          takeSnapshot = true;
          break;

          /*
        case FCam::Event::FocusPressed: // Shutter pressed half-way
          break;
        case FCam::Event::FocusReleased: // Shutter released from being pressed half-way
          break;
          */
        case FCam::Event::ShutterPressed:
          takeSnapshot = true;
          break;
          /*
        case FCam::Event::ShutterReleased:
          takeSnapshot = false;
          break;
          */
        default:
          break;
      }
    }

    /* Take a snapshot if requested. However,
     * If you are running an autofocus algorithm, you might
     * want to wait for it to finish.
     */
    // TODO(edluong): takeSnapshot bool is a bit unnecessary
    if (takeSnapshot) {

      m_snapshot.exposure = m_shot.exposure;
      m_snapshot.frameTime = m_shot.frameTime;
      m_snapshot.gain = m_shot.gain;
      /*
      m_snapshot.actions.clear();
      // Flash if necessary
      if (m_snapshot.gain > 2) {
        fprintf(stderr, "MyCameraThread : Flash required.\n");
        FCam::Flash::FireAction fire(&flash);
        fire.time = m_snapshot.exposure - 30000;
        fire.duration = 30000;
        fire.brightness = flash.maxBrightness();
        m_snapshot.actions.insert(&fire);

        // Reduce gain. The factor of 2 is arbitrary.
        m_snapshot.gain /= 2;
      }      
      */

      if (m_shotMode == HotSpot) {
        scheduledShot = e.iData;
        if (scheduledShot == 0) {
          sensor.capture(m_snapshot);
        }
        else {
          sensor.capture(m_shot);
        }
      }
      else { // Stream
        numStreamed = 0;
        sensor.capture(m_snapshot);
      }
      takeSnapshot = false;
    }

    // This class represents a pointer to a frame given by the sensor.
    FCam::Frame::Ptr f;

    bool gotOne = false;

    /* The sensor may have generated a number of frames in the
     * stream, so we make sure to consume them all and keep the buffer empty.
     */
    do {
      // This method blocks until a frame is obtained
      f = sensor.getFrame();
      int shotId = f->shot().id;

      /* Check how the frame was requested. 
       * One can check this by calling shot().id.
       */
      if (m_shotMode == HotSpot && scheduledShot >= 0) {

        char fname[256];
        snprintf(fname, 255, "./frame_%02d.jpg", scheduledShot);

        if (scheduledShot == 0) {
          if (shotId == m_snapshot.id) {
            if (f->image.valid()) {
              fprintf(stderr, "MyCameraThread: Saving high quality!\n");
              writer.saveJPEG(f, fname, JPEG_QUALITY);
              scheduledShot = -1;
            }
            else {
              fprintf(stderr, "MyCameraThread: Frame dropped!\n");
            }
          }
        }
        else {
          if (f->image.valid()) {
            f->image.lock();
            MySaveJPEG(f, fname, JPEG_QUALITY);
            f->image.unlock();

            unsigned int sharpness = ComputeSharpness(f);
            fprintf(stderr, "sharpness: %zu\n", sharpness);
            scheduledShot = -1;
          }
          else {
            fprintf(stderr, "MyCameraThread: Frame dropped!\n");
          }
        }
      } 
      else if (m_shotMode == Stream && numStreamed >= 0) {
        if (shotId == m_snapshot.id) {
          if (f->image.valid()) {
            char fname[256];
            snprintf(fname, 255, "./frame_stream_%02d.jpg", 0);

            fprintf(stderr, "MyCameraThread: Saving high quality!\n");
            writer.saveJPEG(f, fname, JPEG_QUALITY);
          }
          else {
            fprintf(stderr, "MyCameraThread: Frame dropped!\n");
          }
        }
        else if (!gotOne && numStreamed < kMaxNumStream) {
          if (f->image.valid()) {
            gotOne = true;
            numStreamed++;
            char fname[256];
            snprintf(fname, 255, "./frame_stream_%02d.jpg", numStreamed);

            // apparantly don't need to lock here?
            f->image.lock();
            MySaveJPEG(f, fname, JPEG_QUALITY);
            f->image.unlock();

            m_infowidget->processFrame(f);
            m_viewfinder->update();

            unsigned int sharpness = ComputeSharpness(f);
            fprintf(stderr, "sharpness: %zu\n", sharpness);
          }
          else {
            fprintf(stderr, "MyCameraThread: Frame dropped!\n");
          }
        }
        if (numStreamed >= kMaxNumStream) {
          // done streaming
          numStreamed = -1;
        }
      }
    } while (sensor.framesPending());

    // Make sure that the frame we obtained is actually requested, not garbage.
    if (f->shot().id == m_shot.id) {

      // Ensure that the image is valid.
      if (f->image.valid()) {

        // Lock the image to prevent update while we are using it.
        // If you are only going to read from it, you do not have to lock.
        //	f->image.lock();

        // Tell the infowidget to update.
        m_infowidget->processFrame(f);

        // Tell the viewfinder to update.
        m_viewfinder->update();


        // Unlock the image.
        // f->image.unlock();
      }
    }
  }

  /* Tell the sensor that no more frames are needed. */
  sensor.stopStreaming();
  fprintf(stderr, "MyCameraThread : exiting streaming loop.\n");
}

int MyCameraThread::exposureChanged(int level)  {
  fprintf(stderr, "MycameraThread : recomputing exposure.\n");

  // Cap the value to be between 1 and 50
  level = (level > 50 ? 50 : (level < 0 ? 0 : level));

  const int min_exposure = 35000; // in microseconds
  const int max_exposure = 125000;
  const float max_gain = 32.0f;   
  float max_ratio  = max_exposure * max_gain / min_exposure;

  int desired_exposure  = exp(log(max_ratio)/50.0f*level) * min_exposure;
  if (desired_exposure <= max_exposure) {
    // Use the lowest gain.
    m_shot.exposure = desired_exposure;      
    m_shot.frameTime = desired_exposure;
    m_shot.gain = 1.0f;
  } else {
    // Adjust gain until desired exposure is achieved.
    m_shot.exposure = max_exposure;      
    m_shot.frameTime = max_exposure;
    m_shot.gain = ((float)desired_exposure) / max_exposure;
  }
  /* Note that changing m_shot does not cause the sensor
   * to use the new parameters. You must explicitly tell the sensor
   * to do so. This is done in the main loop with m_exposureUpdated flag.
   */
  m_exposureUpdated = true;
  return m_shot.exposure;
}
