#include "MyCamera.h"

#include "MyCameraThread.h"
#include "MyInfoWidget.h"
#include "MyViewfinderX.h"

MyCamera::MyCamera(QWidget * parent) : QWidget(parent) {
  
  /*
   * The widget will have the following layout
   * +============+====+
   * +            +Far +
   * +            +Near+
   * +            +====+
   * +            + S  +
   * +            + l  +
   * +            + i  +
   * + Viewfinder + d  +
   * +            + e  +
   * +            + r  +
   * +            +====+
   * +            +EXIT+
   * +============+====+
   */

  // Buttons
  m_exit = new QPushButton("Exit", this);
  m_exit->setFixedSize(100,50);
  m_mode_spot = new QPushButton("Spot", this);
  m_mode_spot->setFixedSize(100,50);
  m_mode_stream = new QPushButton("Video", this);
  m_mode_stream->setFixedSize(100,50);

  // Slider for changing exposure
  m_exposure_slider = new QSlider(Qt::Vertical);
  m_exposure_slider->setMinimum(0);
  m_exposure_slider->setMaximum(50);

  // Viewfinder (display raw video)
  MyViewfinderX * viewfinder = new MyViewfinderX(this);
  viewfinder->setFixedSize(640, 480);

  int numRings = 2;
  int numSmallHotSpotsPerRing = 8;

  // InfoWidget (display OpenGL ES)
  MyInfoWidget * infowidget = new MyInfoWidget(viewfinder, HotSpot, numRings, numSmallHotSpotsPerRing);
  infowidget->setGeometry(QRect(1, 1, 638, 478));
  infowidget->setAttribute(Qt::WA_TransparentForMouseEvents);

  // Sublayout containing the widgets to the right.
  QVBoxLayout * sublayout = new QVBoxLayout;
  sublayout->addWidget(m_mode_stream);
  sublayout->addWidget(m_mode_spot);
  sublayout->addWidget(m_exposure_slider);
  sublayout->addWidget(m_exit);

  // Main layout.
  QHBoxLayout * layout = new QHBoxLayout;
  layout->setContentsMargins(0,0,0,0);
  layout->addWidget(viewfinder);
  layout->addLayout(sublayout);
  this->setLayout(layout);

  // Set up camera thread, but do not run it yet.
  m_thread = new MyCameraThread(viewfinder, infowidget, this);

  /* Hook up signals and slots. This is how
   * Qt performs event handling. Whenever a signal is generated,
   * the corresponding slot function is run by the Qt thread (main thread.)
   * In general, all visual aspects (repainting and drawing) should be done
   * this thread, so keep that in mind.
   */
  QObject::connect(m_exit, SIGNAL(clicked()),
		   this, SLOT(scheduleTermination()), Qt::DirectConnection);
  QObject::connect(m_mode_spot, SIGNAL(clicked()), m_thread, SLOT(setShotModeSpot()));
  QObject::connect(m_mode_stream,  SIGNAL(clicked()), m_thread, SLOT(setShotModeStream()));
  QObject::connect(m_exposure_slider, SIGNAL(valueChanged(int)),
		   m_thread, SLOT(exposureChanged(int)), Qt::DirectConnection);

  QObject::connect(viewfinder, SIGNAL(mouseMoved(int, int)),
      infowidget, SLOT(viewfinderMouseMove(int, int)));
  
  // Set exposure to the lowest.
  m_exposure_slider->setValue(1);
}

void 
MyCamera::scheduleTermination() 
{
  // Make sure that the camera thread has finished.
  if (m_thread->isRunning()) {
    m_thread->scheduleTermination();
    m_thread->wait();
  }
  QApplication::exit(0);
}

void MyCamera::startStreaming() {
  if (!m_thread->isRunning())
    m_thread->start();
}


MyCamera::~MyCamera() {
  // Ensure that the thread is not running.
  if (m_thread->isRunning()) {
    m_thread->scheduleTermination();
    m_thread->wait();
  }
}
