#ifndef __RTREYES_CORE_MATH_GEOM_H__
#define __RTREYES_CORE_MATH_GEOM_H__

#include <assert.h>
#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string>
#include <iostream>

#ifdef WIN32

#ifndef M_PI
  #define M_PI 3.14159265
#endif

#pragma warning(disable : 4244 4146)
#endif

// amount squared to figure if two floats are equal
#define VERY_SMALL 1e-6
#define VERY_BIG   1e6

#define VERY_BIG_INT ((int)0x7fffffff)

#define DEGRAD (180.0/M_PI) // degrees per radian
#define RADDEG (M_PI/180.0) // radians per degree

#ifndef MIN
#define MIN(x,y) ( ((x)<=(y)) ? (x) : (y) )
#endif

#ifndef MAX
#define MAX(x,y) ((x) >= (y) ? (x) : (y))
#endif

inline int IntMod4(int x) { return x % 4; }

template <class T>
inline T
Lerp(const T& P0, const T& P1, float s)
{
  return (1.f - s) * P0 + s * P1;
}

inline bool
IsPowerOfTwo(int x) {
  return ((x & (x - 1)) == 0);
}

inline bool
IsEven(int x) {
  return !(x & 1);
}

// Assumes the input is a power of two
inline int
LogPowerOfTwo(int x) {

  assert(IsPowerOfTwo(x));

  int logValue = 0;
  int value = 1;
  while (value != x) {
    logValue++;
    value *= 2;
  }
  return logValue;
}

inline unsigned int
RoundUp16(unsigned int x) {
  return ((x + 15) >> 4) << 4;
}

// 3x3 matrix determinant -- helper function
inline float det3 (float a, float b, float c,
                   float d, float e, float f,
                   float g, float h, float i)
{ return a*e*i + d*h*c + g*b*f - g*e*c - d*b*i - a*h*f; }

class Vec2f
{
 public:
    float p[2];
    inline Vec2f() { /*p[0]=0.0; p[1]=0.0;*/ }
    inline Vec2f(float x, float y) { p[0]=x; p[1]=y; }
    inline Vec2f(const float *f) { p[0]=f[0]; p[1]=f[1]; }
    inline Vec2f(const double *d) { p[0]=d[0]; p[1]=d[1]; }
    inline Vec2f(const Vec2f& v) { p[0]=v.p[0]; p[1]=v.p[1]; }
    inline void setValue(float x, float y) { p[0]=x; p[1]=y; }
    inline float* getValue() { return p; }
    inline const float* getValue() const { return p; }
    inline void set(float *f) { p[0]=f[0]; p[1]=f[1]; }
    inline void scale(float s) { p[0]*=s; p[1]*=s; }
    inline float lengthSquared() const { return p[0]*p[0] + p[1]*p[1]; }
    inline float length() const { return sqrt(p[0]*p[0]+p[1]*p[1]); }
    inline void normalize() { float l = length(); scale(1.f/l); }
    inline Vec2f normalized() const
    { float l = length(); return Vec2f(p[0]/l,p[1]/l); }
    inline void setLength(float l) { scale(l/length()); }
    inline void negate() { p[0]=-p[0]; p[1]=-p[1]; }
    inline Vec2f perp() { return Vec2f(-p[1],p[0]); }
    inline float& operator[](int n)
    { return p[n]; }
    inline const float& operator[](int n) const
    { return p[n]; }
    inline Vec2f operator+(const Vec2f &v) const
    { return Vec2f(p[0]+v.p[0], p[1]+v.p[1]); }
    inline Vec2f operator-(const Vec2f &v) const
    { return Vec2f(p[0]-v.p[0], p[1]-v.p[1]); }
    inline Vec2f operator-() const
    { return Vec2f(-p[0], -p[1]); }
    inline Vec2f operator*(float f) const
    { return Vec2f(p[0]*f, p[1]*f); }
    inline Vec2f operator/(float f) const
    { return Vec2f(p[0]/f, p[1]/f); }
    inline Vec2f& operator+=(const Vec2f& v)
    { p[0]+=v.p[0]; p[1]+=v.p[1]; return *this; }
    inline Vec2f& operator*=(float f)
    { p[0]*=f; p[1]*=f; return *this; }
    inline Vec2f& operator/=(float f)
    { p[0]/=f; p[1]/=f; return *this; }
    inline bool operator==(const Vec2f &v) const
    { return fabs(p[0]-v.p[0])<VERY_SMALL && fabs(p[1]-v.p[1])<VERY_SMALL; }
    inline bool equals(const Vec2f &v, float tol) const
    { return fabs(p[0]-v.p[0])<tol && fabs(p[1]-v.p[1])<tol; }
    inline float dot(const Vec2f &v) const
    { return p[0]*v.p[0] + p[1]*v.p[1]; }
    inline float cross(const Vec2f &v) {
    return p[0]*v.p[1] - p[1]*v.p[0];
  }
};

class Vec2i
{
 public:
    int p[2];
    inline Vec2i() { /*p[0]=0.0; p[1]=0.0;*/ }
    inline Vec2i(int x, int y) { p[0]=x; p[1]=y; }
    inline Vec2i(const int *f) { p[0]=f[0]; p[1]=f[1]; }
    inline Vec2i(const Vec2i& v) { p[0]=v.p[0]; p[1]=v.p[1]; }
    inline void setValue(int x, int y) { p[0]=x; p[1]=y; }
    inline int* getValue() { return p; }
    inline const int* getValue() const { return p; }
    inline void set(int *f) { p[0]=f[0]; p[1]=f[1]; }
    inline void scale(int s) { p[0]*=s; p[1]*=s; }
    inline void negate() { p[0]=-p[0]; p[1]=-p[1]; }
    inline int& operator[](int n)
    { return p[n]; }
    inline const int& operator[](int n) const
    { return p[n]; }
    inline Vec2i operator+(const Vec2i &v) const
    { return Vec2i(p[0]+v.p[0], p[1]+v.p[1]); }
    inline Vec2i operator-(const Vec2i &v) const
    { return Vec2i(p[0]-v.p[0], p[1]-v.p[1]); }
    inline Vec2i operator-() const
    { return Vec2i(-p[0], -p[1]); }
    inline Vec2i operator*(int f) const
    { return Vec2i(p[0]*f, p[1]*f); }
    inline Vec2i operator/(int f) const
    { return Vec2i(p[0]/f, p[1]/f); }
    inline Vec2i& operator+=(const Vec2i& v)
    { p[0]+=v.p[0]; p[1]+=v.p[1]; return *this; }
    inline Vec2i& operator*=(int f)
    { p[0]*=f; p[1]*=f; return *this; }
    inline Vec2i& operator/=(int f)
    { p[0]/=f; p[1]/=f; return *this; }
    inline bool operator==(const Vec2i &v) const
    { return p[0]-v.p[0] == 0 && p[1]-v.p[1] == 0; }
    inline int dot(const Vec2i &v) const
    { return p[0]*v.p[0] + p[1]*v.p[1]; }
    inline int cross(const Vec2i &v) {
    return p[0]*v.p[1] - p[1]*v.p[0];
  }
};

class Vec3f
{
 public:
    float p[3];
  inline Vec3f() { /*p[0]=0.0; p[1]=0.0; p[2]=0.0;*/}
    inline Vec3f(float x, float y, float z) { p[0]=x; p[1]=y; p[2]=z; }
    inline Vec3f(const float *f) { p[0]=f[0]; p[1]=f[1]; p[2]=f[2]; }
    inline Vec3f(const double *d) { p[0]=d[0]; p[1]=d[1]; p[2]=d[2]; }
    inline Vec3f(const Vec3f& v) { p[0]=v.p[0]; p[1]=v.p[1]; p[2]=v.p[2];}
    inline void setValue(float x, float y, float z) { p[0]=x; p[1]=y; p[2]=z; }
    inline float* getValue() { return p; }
    inline const float* getValue() const { return p; }
    inline void set(float *f) { p[0]=f[0]; p[1]=f[1]; p[2]=f[2]; }
    inline void scale(float s) { p[0]*=s; p[1]*=s; p[2]*=s; }
    inline float length() const { return sqrtf(p[0]*p[0]+p[1]*p[1]+p[2]*p[2]); }
    inline void normalize() { float l = length(); scale(1.0/l); }
    inline Vec3f normalized() const
    { float l = length(); return Vec3f(p[0]/l,p[1]/l,p[2]/l); }
    inline void setLength(float l) { scale(l/length()); }
    inline void negate() { p[0]=-p[0]; p[1]=-p[1]; p[2]=-p[2]; }
    inline float& operator[](int n)
    { return p[n]; }
    inline const float& operator[](int n) const
    { return p[n]; }
    inline Vec3f operator+(const Vec3f &v) const
    { return Vec3f(p[0]+v.p[0], p[1]+v.p[1], p[2]+v.p[2]); }
    inline Vec3f operator-(const Vec3f &v) const
    { return Vec3f(p[0]-v.p[0], p[1]-v.p[1], p[2]-v.p[2]); }
    inline Vec3f operator-() const
    { return Vec3f(-p[0], -p[1], -p[2]); }
    inline Vec3f operator*(float f) const
    { return Vec3f(p[0]*f, p[1]*f, p[2]*f); }
    inline Vec3f operator*(const Vec3f& rhs) const
    { return Vec3f(p[0]*rhs[0], p[1]*rhs[1], p[2]*rhs[2]); }
    inline Vec3f operator/(float f) const
    { return Vec3f(p[0]/f, p[1]/f, p[2]/f); }
    inline Vec3f& operator+=(const Vec3f& v)
    { p[0]+=v.p[0]; p[1]+=v.p[1]; p[2]+=v.p[2]; return *this; }
    inline Vec3f& operator-=(const Vec3f& v)
    { p[0]-=v.p[0]; p[1]-=v.p[1]; p[2]-=v.p[2]; return *this; }
    inline Vec3f& operator*=(float f)
    { p[0]*=f; p[1]*=f; p[2]*=f; return *this; }
    inline Vec3f& operator*=(const Vec3f& rhs)
    { p[0]*=rhs[0]; p[1]*=rhs[1]; p[2]*=rhs[2]; return *this; }
    inline Vec3f& operator/=(float f)
    { p[0]/=f; p[1]/=f; p[2]/=f; return *this; }
    inline bool operator==(const Vec3f &v) const
    { return fabs(p[0]-v.p[0])<VERY_SMALL
            && fabs(p[1]-v.p[1])<VERY_SMALL
            && fabs(p[2]-v.p[2])<VERY_SMALL; }
    inline bool equals(const Vec3f &v, float tol) const
    { return fabs(p[0]-v.p[0])<tol
            && fabs(p[1]-v.p[1])<tol
            && fabs(p[2]-v.p[2])<tol; }
    inline float dot(const Vec3f &v) const
    { return p[0]*v.p[0] + p[1]*v.p[1] + p[2]*v.p[2]; }
    inline Vec3f cross(const Vec3f &v) const
    { return Vec3f(p[1]*v.p[2] - p[2]*v.p[1],
       p[2]*v.p[0] - p[0]*v.p[2],
       p[0]*v.p[1] - p[1]*v.p[0]); }
    void rotateXY(float theta, float tx, float ty);
    inline int maxAbsoluteComponent() const {
      float abs_x = fabsf(p[0]);
      float abs_y = fabsf(p[1]);
      float abs_z = fabsf(p[2]);
      if (abs_x > abs_y) {
        return (abs_x > abs_z) ? 0 : 2;
      } else {
        return (abs_y > abs_z) ? 1 : 2;
      }
    }
};

inline std::istream& operator>>(std::istream& is, Vec3f& v) {
  is >> v[0] >> v[1] >> v[2];
  return is;
}

inline std::ostream& operator<<(std::ostream& os, const Vec3f& v) {
  os << v[0] << " " << v[1] << " " << v[2];
  return os;
}

inline Vec2f operator*(float t, const Vec2f& v) {
  return Vec2f(t * v.p[0], t * v.p[1]);
}

inline Vec3f operator*(float f, const Vec3f &v)
{
  return Vec3f(v[0]*f, v[1]*f, v[2]*f);
}

class Vec3i
{
 public:
    int p[3];
  Vec3i() { /*p[0]=0; p[1]=0; p[2]=0;*/ }
    Vec3i(int x, int y, int z)
    { p[0]=x; p[1]=y; p[2]=z; }
    Vec3i(const int *f) { p[0]=f[0]; p[1]=f[1]; p[2]=f[2]; }
    Vec3i(const Vec3i& v) { p[0]=v.p[0]; p[1]=v.p[1]; p[2]=v.p[2];}
    inline void setValue(int x, int y, int z) { p[0]=x; p[1]=y; p[2]=z; }
    inline int* getValue() { return p; }
    inline const int* getValue() const { return p; }
    inline void set(int *f) { p[0]=f[0]; p[1]=f[1]; p[2]=f[2]; }
    inline void scale(int s) { p[0]*=s; p[1]*=s; p[2]*=s; }
    inline void negate() { p[0]=-p[0]; p[1]=-p[1]; p[2]=-p[2]; }
    inline int& operator[](int n)
    { return p[n]; }
    inline const int& operator[](int n) const
    { return p[n]; }
    Vec3i operator+(const Vec3i &v)
    { return Vec3i(p[0]+v.p[0], p[1]+v.p[1], p[2]+v.p[2]); }
    Vec3i operator-(const Vec3i &v)
    { return Vec3i(p[0]-v.p[0], p[1]-v.p[1], p[2]-v.p[2]); }
    Vec3i operator-()
    { return Vec3i(-p[0], -p[1], -p[2]); }
    Vec3i operator*(int f)
    { return Vec3i(p[0]*f, p[1]*f, p[2]*f); }
    Vec3i operator/(int f)
    { return Vec3i(p[0]/f, p[1]/f, p[2]/f); }
    Vec3i& operator+=(const Vec3i& v)
    { p[0]+=v.p[0]; p[1]+=v.p[1]; p[2]+=v.p[2]; return *this; }
    Vec3i& operator*=(int f)
    { p[0]*=f; p[1]*=f; p[2]*=f; return *this; }
    Vec3i& operator/=(int f)
    { p[0]/=f; p[1]/=f; p[2]/=f; return *this; }
    bool operator==(const Vec3i &v)
    { return p[0]==v.p[0] && p[1]==v.p[1] && p[2]==v.p[2]; }
};

struct BBox2f {
  Vec2f bmin, bmax;
  BBox2f() {
    bmin[0] = bmin[1] = VERY_BIG;
    bmax[0] = bmax[1] = -VERY_BIG;
  }
  BBox2f(const Vec2f& a, const Vec2f& b) : bmin(a), bmax(b) {
  }

  void Reset() {
    bmin[0] = bmin[1] = VERY_BIG;
    bmax[0] = bmax[1] = -VERY_BIG;
  }

  void AddPoint(const Vec2f& a) {
    if (a[0] < bmin[0]) bmin[0] = a[0];
    if (a[0] > bmax[0]) bmax[0] = a[0];
    if (a[1] < bmin[1]) bmin[1] = a[1];
    if (a[1] > bmax[1]) bmax[1] = a[1];
  }

  void AddBox(const BBox2f& b) {
    for (int i = 0; i < 2; i++) {
      bmin[i] = std::min(bmin[i], b.bmin[i]);
      bmax[i] = std::max(bmax[i], b.bmax[i]);
    }
  }

  void Expand(float xExpand, float yExpand) {
    bmin[0] -= xExpand;
    bmax[0] += xExpand;
    bmin[1] -= yExpand;
    bmax[1] += yExpand;
  }

  bool Overlap(const BBox2f& bbox) {
    return !(bmin[0] > bbox.bmax[0] ||
             bmin[1] > bbox.bmax[1] ||
             bmax[0] < bbox.bmin[0] ||
             bmax[1] < bbox.bmin[1]);
  }

  bool Inside(const Vec2f& pt) const {
    if (pt[0] < bmin[0] || pt[0] > bmax[0] ||
        pt[1] < bmin[1] || pt[1] > bmax[1])
      return false;
    else
      return true;
  }

  inline Vec2f Centroid() const {
    return 0.5f * (bmin + bmax);
  }
};

struct BBox2i {
  int bmin[2];
  int bmax[2];

  BBox2i() {
    Reset();
  }

  inline void Reset() {
    bmin[0] = bmin[1] = (int)VERY_BIG;
    bmax[0] = bmax[1] = (int)-VERY_BIG;
  }

  inline bool Empty() const {
    return (bmin[0] >= bmax[0]) || (bmin[1] >= bmax[1]);
  }
};

class Matrix;

struct BBox3f
{
  Vec3f bmin, bmax;
  inline BBox3f(const Vec3f& a, const Vec3f& b) : bmin(a), bmax(b) {
  }
  inline BBox3f() {
    bmin[0] = bmin[1] = bmin[2] = VERY_BIG;
    bmax[0] = bmax[1] = bmax[2] = -VERY_BIG;
  }
  inline BBox3f(const BBox3f& other) : bmin(other.bmin), bmax(other.bmax) {
  }
  inline BBox3f& operator=(const BBox3f& other) {
    bmin = other.bmin;
    bmax = other.bmax;
    return *this;
  }

  inline void Reset() {
    bmin[0] = bmin[1] = bmin[2] = VERY_BIG;
    bmax[0] = bmax[1] = bmax[2] = -VERY_BIG;
  }

  inline void AddPoint(const Vec3f& pt) {
    for (int i=0; i<3; i++) {
      bmin[i] = std::min(bmin[i], pt[i]);
      bmax[i] = std::max(bmax[i], pt[i]);
    }
  }

  inline Vec3f Corner(int i) const {
    int x_idx = (i & 4) >> 2;
    int y_idx = (i & 2) >> 1;
    int z_idx = (i & 1);
    return Vec3f( x_idx == 0 ? bmin[0] : bmax[0],
      y_idx == 0 ? bmin[1] : bmax[1],
      z_idx == 0 ? bmin[2] : bmax[2]);
  }

  inline float Area() const {
    Vec3f diag = bmax - bmin;
    diag[0] = std::max(0.f, diag[0]);
    diag[1] = std::max(0.f, diag[1]);
    diag[2] = std::max(0.f, diag[2]);
    return 2.f * (diag[0] * diag[1] + diag[1] * diag[2] + diag[0] * diag[2]);
  }

  inline Vec3f Centroid() const {
    return .5f * (bmin + bmax);
  }

  inline Vec3f Diagonal() const {
    return bmax - bmin;
  }

  inline const Vec3f& Min() const {
    return bmin;
  }

  inline const Vec3f& Max() const {
    return bmax;
  }

  inline const Vec3f& MinOrMax(int which) const {
    return (which == 0) ? bmin : bmax;
  }

  inline void AddBox(const BBox3f& b) {
    // NOTE(boulos): Do not use AddPoint!, we want to just consider
    // bmin as expanding the min not the max so that AddBox of the
    // Reset() box doesn't mess things up.
    for (int i = 0; i < 3; i++) {
      bmin[i] = std::min(bmin[i], b.bmin[i]);
      bmax[i] = std::max(bmax[i], b.bmax[i]);
    }
  }

  inline void Expand(float bloat) {
    bmin -= Vec3f(bloat, bloat, bloat);
    bmax += Vec3f(bloat, bloat, bloat);
  }

  inline int LongestAxis() const {
    Vec3f diag = Diagonal();
    if (diag[0] > diag[1]) {
      return (diag[0] > diag[2]) ? 0 : 2;
    }
    return (diag[1] > diag[2]) ? 1 : 2;
  }

  void Transform(const Matrix& m, BBox3f& out) const;

};

class Vec4f
{
 private:
    float p[4];
 public:
    Vec4f() { p[0]=0.0; p[1]=0.0; p[2]=0.0;; p[3]=0.0; }
    Vec4f(float x, float y, float z, float w) { p[0]=x; p[1]=y; p[2]=z; p[3]=w; }
    Vec4f(const float *f) { p[0]=f[0]; p[1]=f[1]; p[2]=f[2]; p[3]=f[3]; }
    Vec4f(const double *d) { p[0]=d[0]; p[1]=d[1]; p[2]=d[2]; p[3]=d[3]; }
    Vec4f(const Vec4f& v) { p[0]=v.p[0]; p[1]=v.p[1]; p[2]=v.p[2]; p[3]=v.p[3];}
    Vec4f(const Vec3f& v, float w) { p[0]=v.p[0]; p[1]=v.p[1]; p[2]=v.p[2]; p[3]=w;}
    inline void setValue(float x, float y, float z, float w) { p[0]=x; p[1]=y; p[2]=z; p[3]=w; }
    inline float* getValue() { return p; }
    inline const float* getValue() const { return p; }
    inline void set(float *f) { p[0]=f[0]; p[1]=f[1]; p[2]=f[2]; p[3]=f[3]; }
    inline void scale(float s) { p[0]*=s; p[1]*=s; p[2]*=s; }
    inline float length() { return sqrt(p[0]*p[0]+p[1]*p[1]+p[2]*p[2]+p[3]*p[3]); }
    inline void normalize() { scale(1.0/length()); }
    inline void setLength(float l) { scale(l/length()); }
    inline void negate() { p[0]=-p[0]; p[1]=-p[1]; p[2]=-p[2]; p[2]=-p[3]; }
    inline float& operator[](int n) { return p[n]; }
    inline const float& operator[](int n) const
    { return p[n]; }
    Vec4f operator+(const Vec4f &v)
    { return Vec4f(p[0]+v.p[0], p[1]+v.p[1], p[2]+v.p[2], p[3]+v.p[3]); }
    Vec4f operator-(const Vec4f &v)
    { return Vec4f(p[0]-v.p[0], p[1]-v.p[1], p[2]-v.p[2], p[3]+v.p[3]); }
    Vec4f operator-()
    { return Vec4f(-p[0], -p[1], -p[2], -p[3]); }
    Vec4f operator*(float f)
    { return Vec4f(p[0]*f, p[1]*f, p[2]*f, p[3]*f); }
    Vec4f operator/(float f)
    { return Vec4f(p[0]/f, p[1]/f, p[2]/f, p[3]/f); }
    Vec4f& operator*=(float f)
    { p[0]*=f; p[1]*=f; p[2]*=f; p[3]*=f; return *this; }
     Vec4f& operator/=(float f)
    { p[0]/=f; p[1]/=f; p[2]/=f; p[3]/=f; return *this; }
  Vec4f& operator+=(const Vec4f& v)
    { p[0]+=v.p[0]; p[1]+=v.p[1]; p[2]+=v.p[2]; p[3]+=v.p[3]; return *this; }
    bool operator==(const Vec4f &v)
    { return fabs(p[0]-v.p[0])<VERY_SMALL
            && fabs(p[1]-v.p[1])<VERY_SMALL
            && fabs(p[2]-v.p[2])<VERY_SMALL
            && fabs(p[3]-v.p[3])<VERY_SMALL; }
    bool equals(const Vec4f &v, float tol)
    { return fabs(p[0]-v.p[0])<tol
            && fabs(p[1]-v.p[1])<tol
            && fabs(p[2]-v.p[2])<tol
            && fabs(p[3]-v.p[3])<tol; }
};

inline Vec4f operator*(float f, const Vec4f &v)
{
  return Vec4f(v[0]*f, v[1]*f, v[2]*f, v[3]*f);
}


/*
 * Column major matrix representation
 */
class Matrix
{
private:
  float matrix[4][4];

public:
    Matrix() {}
  Matrix(const float* data) {
    for (int i = 0; i < 4; i++) {
      for (int j = 0; j < 4; j++) {
        matrix[i][j] = data[4*i+j];
      }
    }
  }
  Matrix(const float data[4][4]) {
    for (int i = 0; i < 4; i++) {
      for (int j = 0; j < 4; j++) {
        matrix[i][j] = data[i][j];
      }
    }
  }

    inline float* operator [](int i) { return &matrix[i][0]; }
    inline const float* operator [](int i) const  { return &matrix[i][0]; }

  Matrix& operator*=(float f) {
    for (int i=0; i<4; i++)
      for (int j=0; j<4; j++)
        matrix[i][j] *= f;
    return *this;
  }

  void print() const;
  void WriteToRIB(FILE* output, std::string indent) const;

  void Transform(const Vec4f& src, Vec4f& dst) const;
  void TransformPoint(const Vec3f &src, Vec3f &dst) const;
  void TransformPoint(const Vec3f &src, Vec4f &dst) const;
  void TransposeTransformPoint(const Vec3f& src, Vec3f& dst) const;
  void TransformVector(const Vec3f &src, Vec3f &dst) const;
  void TransposeTransformVector(const Vec3f &src, Vec3f &dst) const;

  void multMatrix(const Matrix &src, Matrix &dst) const;
  void transpose(Matrix& dst) const;
  void Inverse(Matrix& inv) const;
  float Determinant() const;
  float Norm() const {
    // Compute the Frobenius norm for now
    float sum_squares = 0.f;
    for (int i = 0; i < 3; i++) {
      for (int j = 0; j < 3; j++) {
        sum_squares += (matrix[i][j] * matrix[i][j]);
      }
    }
    return sqrtf(sum_squares);
  }

  static void Identity(Matrix& mat);
  static void Translate(Matrix& mat, float x, float y, float z);
  static void Scale(Matrix& mat, float x, float y, float z);
  static void RotateX(Matrix& mat, float rad);
  static void RotateY(Matrix& mat, float rad);
  static void RotateZ(Matrix& mat, float rad);
  static void RotateAxis(Matrix& mat, const Vec3f& axis, float rad);
  static void Lookat(Matrix& mat, const Vec3f& eye, const Vec3f& at, const Vec3f& up);
  static void LookatInv(Matrix& mat, const Vec3f& eye, const Vec3f& at, const Vec3f& up);
};

class Transform {
public:
  Matrix m;
  Matrix mInv;

  void mult(const Transform& t, Transform& result) const;

  static void Identity(Transform& t);
  static void Translate(Transform& t, float x, float y, float z);
  static void Scale(Transform& t, float x, float y, float z);
  static void RotateX(Transform& t, float rad);
  static void RotateY(Transform& t, float rad);
  static void RotateZ(Transform& t, float rad);
  static void RotateAxis(Transform& t, const Vec3f& axis, float rad);
  static void SetMatrix(Transform& t, const Matrix& m);
};

class Rotation
{
 public:

    // Default constructor
    Rotation() {}

    // Constructor given a quaternion as an array of 4 components
    Rotation(const float v[4]) { setValue(v); }

    // Constructor given 4 individual components of a quaternion
    Rotation(float q0, float q1, float q2, float q3)
  { setValue(q0, q1, q2, q3); }

    // Constructor given a rotation matrix
    Rotation(const Matrix &m) { setValue(m); }

    // Constructor given 3D rotation axis vector and angle in radians
    Rotation(const Vec3f &axis, float radians)
  { setValue(axis, radians); }

    // Constructor for rotation that rotates one direction vector to another
    Rotation(const Vec3f &rotateFrom, const Vec3f &rotateTo)
  { setValue(rotateFrom, rotateTo); }

    // Returns pointer to array of 4 components defining quaternion
    const float * getValue() const { return (quat); }

    // Returns 4 individual components of rotation quaternion
    void getValue(float &q0, float &q1, float &q2, float &q3) const;

    // Returns corresponding 3D rotation axis vector and angle in radians
    void getValue(Vec3f &axis, float &radians) const;

    // Returns corresponding 4x4 rotation matrix
    void getValue(Matrix &matrix) const;

    // Changes a rotation to be its inverse
    Rotation& invert();

    // Returns the inverse of a rotation
    Rotation inverse() const { Rotation q = *this; return q.invert(); }

    // Sets value of rotation from array of 4 components of a quaternion
    Rotation& setValue(const float q[4]);

    // Sets value of rotation from 4 individual components of a quaternion
    Rotation& setValue(float q0, float q1, float q2, float q3);

    // Sets value of rotation from a rotation matrix
    Rotation& setValue(const Matrix &m);

    // Sets value of vector from 3D rotation axis vector and angle in radians
    Rotation& setValue(const Vec3f &axis, float radians);

    // Sets rotation to rotate one direction vector to another
    Rotation& setValue(const Vec3f &rotateFrom, const Vec3f &rotateTo);

    // Multiplies by another rotation; results in product of rotations
    Rotation& operator *=(const Rotation &q);

    // Equality comparison operator
    friend int operator ==(const Rotation &q1, const Rotation &q2);
    friend int operator !=(const Rotation &q1, const Rotation &q2)
  { return !(q1 == q2); }

    // Equality comparison within given tolerance - the square of the
    // length of the maximum distance between the two quaternion vectors
    bool equals(const Rotation &r, float tolerance) const;

    // Multiplication of two rotations; results in product of rotations
    friend Rotation operator *(const Rotation &q1, const Rotation &q2);

    // Puts the given vector through this rotation
    // (Multiplies the given vector by the matrix of this rotation),.
    void multVec(const Vec3f &src, Vec3f &dst) const;

    // Keep the axis the same. Multiply the angle of rotation by
    // the amount 'scaleFactor'
    void scaleAngle( float scaleFactor );

    // Spherical linear interpolation: as t goes from 0 to 1, returned
    // value goes from rot0 to rot1
    static Rotation slerp(const Rotation &rot0, const Rotation &rot1, float t);

    // Null rotation
    static Rotation identity()
  { return Rotation(0.0, 0.0, 0.0, 1.0); }

 private:
    float quat[4]; // Storage for quaternion components

    // Returns the norm (square of the 4D length) of a rotation's quaterion
    float norm() const;

    // Normalizes a rotation quaternion to unit 4D length
    void normalize();
};

inline static float
ComputeTriangleArea(float a, float b, float c) {
  float s = 0.5f * (a + b + c);
  float triangleArea = sqrtf(s * (s - a) * (s - b) * (s - c) );
  if (triangleArea != triangleArea || triangleArea < 0.0f)
    triangleArea = 0.0f;

  return triangleArea;
}

inline static
bool IntersectBox(const BBox3f& box,
                  const Vec3f& org,
                  const Vec3f& rcp,
                  const float tinit,
                  const int signs[3]) {
  float tmin = 0.f;
  float tmax = tinit;
  {
    int x_sign = signs[0];
    float x_min = (box.MinOrMax(  x_sign)[0] - org[0]) * rcp[0];
    float x_max = (box.MinOrMax(1-x_sign)[0] - org[0]) * rcp[0];
    tmin = std::max(x_min, tmin);
    tmax = std::min(x_max, tmax);
  }
  {
    int y_sign = signs[1];
    float y_min = (box.MinOrMax(  y_sign)[1] - org[1]) * rcp[1];
    float y_max = (box.MinOrMax(1-y_sign)[1] - org[1]) * rcp[1];
    tmin = std::max(y_min, tmin);
    tmax = std::min(y_max, tmax);
  }
  {
    int z_sign = signs[2];
    float z_min = (box.MinOrMax(  z_sign)[2] - org[2]) * rcp[2];
    float z_max = (box.MinOrMax(1-z_sign)[2] - org[2]) * rcp[2];
    tmin = std::max(z_min, tmin);
    tmax = std::min(z_max, tmax);
  }
  return tmin <= tmax;
}

inline int BestSeparatingAxis(const BBox3f& left_bounds, const BBox3f& right_bounds, bool& needs_swap) {
  Vec3f left_centroid = left_bounds.Centroid();
  Vec3f right_centroid = right_bounds.Centroid();
  Vec3f diff = right_centroid - left_centroid;
  int best_axis = diff.maxAbsoluteComponent();
  needs_swap = (diff[best_axis] < 0.f);
  return best_axis;
}

#endif

