#include "geom.h"

#include <stdio.h>

void Vec3f::rotateXY(float theta, float tx, float ty)
{
  float costh = cos(theta*RADDEG);
  float sinth = sin(theta*RADDEG);
  float x = p[0]*costh-p[1]*sinth+(-tx*costh+ty*sinth+tx);
  float y = p[0]*sinth+p[1]*costh+(-tx*sinth-ty*costh+ty);
  p[0] = x;
  p[1] = y;
}

/*
 * row-vector post multiplication (WITH A HOMOGENEOUS 4-vec)
 */
void Matrix::Transform(const Vec4f &src, Vec4f &dst) const
{
  float x,y,z,w;

  x = src[0]*matrix[0][0] + src[1]*matrix[1][0] +
    src[2]*matrix[2][0] + src[3]*matrix[3][0];
  y = src[0]*matrix[0][1] + src[1]*matrix[1][1] +
    src[2]*matrix[2][1] + src[3]*matrix[3][1];
  z = src[0]*matrix[0][2] + src[1]*matrix[1][2] +
    src[2]*matrix[2][2] + src[3]*matrix[3][2];
  w = src[0]*matrix[0][3] + src[1]*matrix[1][3] +
    src[2]*matrix[2][3] + src[3]*matrix[3][3];

  dst.setValue(x, y, z, w);
}

/*
 * row-vector post multiplication (WITH A POINT)
 */
void Matrix::TransformPoint(const Vec3f &src, Vec3f &dst) const
{
  float x,y,z,w;

  x = src[0]*matrix[0][0] + src[1]*matrix[1][0] +
    src[2]*matrix[2][0] + matrix[3][0];
  y = src[0]*matrix[0][1] + src[1]*matrix[1][1] +
    src[2]*matrix[2][1] + matrix[3][1];
  z = src[0]*matrix[0][2] + src[1]*matrix[1][2] +
    src[2]*matrix[2][2] + matrix[3][2];
  w = src[0]*matrix[0][3] + src[1]*matrix[1][3] +
    src[2]*matrix[2][3] + matrix[3][3];

  dst.setValue(x/w, y/w, z/w);
}

/*
 * row-vector post multiplication  (WITH A POINT. HOMOGENEOUS OUTPUT)
 */
void Matrix::TransformPoint(const Vec3f &src, Vec4f &dst) const
{
  float x,y,z,w;

  x = src[0]*matrix[0][0] + src[1]*matrix[1][0] +
    src[2]*matrix[2][0] + matrix[3][0];
  y = src[0]*matrix[0][1] + src[1]*matrix[1][1] +
    src[2]*matrix[2][1] + matrix[3][1];
  z = src[0]*matrix[0][2] + src[1]*matrix[1][2] +
    src[2]*matrix[2][2] + matrix[3][2];
  w = src[0]*matrix[0][3] + src[1]*matrix[1][3] +
    src[2]*matrix[2][3] + matrix[3][3];

  dst.setValue(x, y, z, w);
}

/*
 * row-vector post multiplication with transpose of matrix (so it's
 * the same as premultiplication)  (WITH A POINT)
 */
void
Matrix::TransposeTransformPoint(const Vec3f& src, Vec3f& dst) const {
  float x,y,z,w;
  x = src[0]*matrix[0][0] + src[1]*matrix[0][1] +
    src[2]*matrix[0][2] + matrix[0][3];
  y = src[0]*matrix[1][0] + src[1]*matrix[1][1] +
    src[2]*matrix[1][2] + matrix[1][3];
  z = src[0]*matrix[2][0] + src[1]*matrix[2][1] +
    src[2]*matrix[2][2] + matrix[2][3];
  w = src[0]*matrix[3][0] + src[1]*matrix[3][1] +
    src[2]*matrix[3][2] + matrix[3][3];
  dst.setValue(x/w, y/w, z/w);
}

/*
 * row-vector post multiplication, but treat the vector as a normal
 * with w component = 0.  Note that the matrix cannot be any sort of
 * projection.  (WITH A VECTOR)
 */
void Matrix::TransformVector(const Vec3f &src, Vec3f &dst) const
{
  float x,y,z;

  x = src[0]*matrix[0][0] + src[1]*matrix[1][0] +
    src[2]*matrix[2][0];
  y = src[0]*matrix[0][1] + src[1]*matrix[1][1] +
    src[2]*matrix[2][1];
  z = src[0]*matrix[0][2] + src[1]*matrix[1][2] +
    src[2]*matrix[2][2];

  dst.setValue(x, y, z);
}

/*
 * row-vector post multiplication with transpose of the matrix, but
 * treat the vector as a normal with w component = 0.  Note that the
 * matrix cannot be any sort of projection.  (WITH A VECTOR)
 */
void Matrix::TransposeTransformVector(const Vec3f &src, Vec3f &dst) const
{
  float x,y,z;

  x = src[0]*matrix[0][0] + src[1]*matrix[0][1] +
    src[2]*matrix[0][2];
  y = src[0]*matrix[1][0] + src[1]*matrix[1][1] +
    src[2]*matrix[1][2];
  z = src[0]*matrix[2][0] + src[1]*matrix[2][1] +
    src[2]*matrix[2][2];

  dst.setValue(x, y, z);
}

void Matrix::multMatrix(const Matrix &src, Matrix &dst) const
{
  const float* a = &matrix[0][0];
  const float* b = src[0];
  float* c = dst[0];

  c[0] = a[0]*b[0] + a[1]*b[4] + a[2]*b[8]  + a[3]*b[12];
  c[1] = a[0]*b[1] + a[1]*b[5] + a[2]*b[9]  + a[3]*b[13];
  c[2] = a[0]*b[2] + a[1]*b[6] + a[2]*b[10] + a[3]*b[14];
  c[3] = a[0]*b[3] + a[1]*b[7] + a[2]*b[11] + a[3]*b[15];

  c[4] = a[4]*b[0] + a[5]*b[4] + a[6]*b[8]  + a[7]*b[12];
  c[5] = a[4]*b[1] + a[5]*b[5] + a[6]*b[9]  + a[7]*b[13];
  c[6] = a[4]*b[2] + a[5]*b[6] + a[6]*b[10] + a[7]*b[14];
  c[7] = a[4]*b[3] + a[5]*b[7] + a[6]*b[11] + a[7]*b[15];

  c[8]  = a[8]*b[0] + a[9]*b[4] + a[10]*b[8]  + a[11]*b[12];
  c[9]  = a[8]*b[1] + a[9]*b[5] + a[10]*b[9]  + a[11]*b[13];
  c[10] = a[8]*b[2] + a[9]*b[6] + a[10]*b[10] + a[11]*b[14];
  c[11] = a[8]*b[3] + a[9]*b[7] + a[10]*b[11] + a[11]*b[15];

  c[12] = a[12]*b[0] + a[13]*b[4] + a[14]*b[8]  + a[15]*b[12];
  c[13] = a[12]*b[1] + a[13]*b[5] + a[14]*b[9]  + a[15]*b[13];
  c[14] = a[12]*b[2] + a[13]*b[6] + a[14]*b[10] + a[15]*b[14];
  c[15] = a[12]*b[3] + a[13]*b[7] + a[14]*b[11] + a[15]*b[15];
}

void
Matrix::transpose(Matrix& dst) const {
  const float* a = &matrix[0][0];
  float* b = dst[0];
  b[0] = a[0];  b[1] = a[4];  b[2] = a[8];   b[3] = a[12];
  b[4] = a[1];  b[5] = a[5];  b[6] = a[9];   b[7] = a[13];
  b[8] = a[2];  b[9] = a[6];  b[10] = a[10]; b[11] = a[14];
  b[12] = a[3]; b[13] = a[7]; b[14] = a[11]; b[15] = a[15];
}

float
Matrix::Determinant() const {
  float det;
  det  = matrix[0][0] * det3(matrix[1][1], matrix[1][2], matrix[1][3],
                             matrix[2][1], matrix[2][2], matrix[2][3],
                             matrix[3][1], matrix[3][2], matrix[3][3]);
  det -= matrix[0][1] * det3(matrix[1][0], matrix[1][2], matrix[1][3],
                             matrix[2][0], matrix[2][2], matrix[2][3],
                             matrix[3][0], matrix[3][2], matrix[3][3]);
  det += matrix[0][2] * det3(matrix[1][0], matrix[1][1], matrix[1][3],
                             matrix[2][0], matrix[2][1], matrix[2][3],
                             matrix[3][0], matrix[3][1], matrix[3][3]);
  det -= matrix[0][3] * det3(matrix[1][0], matrix[1][1], matrix[1][2],
                             matrix[2][0], matrix[2][1], matrix[2][2],
                             matrix[3][0], matrix[3][1], matrix[3][2]);
  return det;
}

// store inverse of matrix into 'inv'
void
Matrix::Inverse(Matrix& inv) const {

  float det = Determinant();

  inv[0][0]  = det3(matrix[1][1], matrix[1][2], matrix[1][3],
                    matrix[2][1], matrix[2][2], matrix[2][3],
                    matrix[3][1], matrix[3][2], matrix[3][3]) / det;
  inv[0][1] = -det3(matrix[0][1], matrix[0][2], matrix[0][3],
                    matrix[2][1], matrix[2][2], matrix[2][3],
                    matrix[3][1], matrix[3][2], matrix[3][3]) / det;
  inv[0][2]  = det3(matrix[0][1], matrix[0][2], matrix[0][3],
                    matrix[1][1], matrix[1][2], matrix[1][3],
                    matrix[3][1], matrix[3][2], matrix[3][3]) / det;
  inv[0][3] = -det3(matrix[0][1], matrix[0][2], matrix[0][3],
                    matrix[1][1], matrix[1][2], matrix[1][3],
                    matrix[2][1], matrix[2][2], matrix[2][3]) / det;

  inv[1][0] = -det3(matrix[1][0], matrix[1][2], matrix[1][3],
                    matrix[2][0], matrix[2][2], matrix[2][3],
                    matrix[3][0], matrix[3][2], matrix[3][3]) / det;
  inv[1][1]  = det3(matrix[0][0], matrix[0][2], matrix[0][3],
                    matrix[2][0], matrix[2][2], matrix[2][3],
                    matrix[3][0], matrix[3][2], matrix[3][3]) / det;
  inv[1][2] = -det3(matrix[0][0], matrix[0][2], matrix[0][3],
                    matrix[1][0], matrix[1][2], matrix[1][3],
                    matrix[3][0], matrix[3][2], matrix[3][3]) / det;
  inv[1][3]  = det3(matrix[0][0], matrix[0][2], matrix[0][3],
                    matrix[1][0], matrix[1][2], matrix[1][3],
                    matrix[2][0], matrix[2][2], matrix[2][3]) / det;

  inv[2][0]  = det3(matrix[1][0], matrix[1][1], matrix[1][3],
                    matrix[2][0], matrix[2][1], matrix[2][3],
                    matrix[3][0], matrix[3][1], matrix[3][3]) / det;
  inv[2][1] = -det3(matrix[0][0], matrix[0][1], matrix[0][3],
                    matrix[2][0], matrix[2][1], matrix[2][3],
                    matrix[3][0], matrix[3][1], matrix[3][3]) / det;
  inv[2][2]  = det3(matrix[0][0], matrix[0][1], matrix[0][3],
                    matrix[1][0], matrix[1][1], matrix[1][3],
                    matrix[3][0], matrix[3][1], matrix[3][3]) / det;
  inv[2][3] = -det3(matrix[0][0], matrix[0][1], matrix[0][3],
                    matrix[1][0], matrix[1][1], matrix[1][3],
                    matrix[2][0], matrix[2][1], matrix[2][3]) / det;

  inv[3][0] = -det3(matrix[1][0], matrix[1][1], matrix[1][2],
                    matrix[2][0], matrix[2][1], matrix[2][2],
                    matrix[3][0], matrix[3][1], matrix[3][2]) / det;
  inv[3][1] =  det3(matrix[0][0], matrix[0][1], matrix[0][2],
                    matrix[2][0], matrix[2][1], matrix[2][2],
                    matrix[3][0], matrix[3][1], matrix[3][2]) / det;
  inv[3][2] = -det3(matrix[0][0], matrix[0][1], matrix[0][2],
                    matrix[1][0], matrix[1][1], matrix[1][2],
                    matrix[3][0], matrix[3][1], matrix[3][2]) / det;
  inv[3][3] =  det3(matrix[0][0], matrix[0][1], matrix[0][2],
                    matrix[1][0], matrix[1][1], matrix[1][2],
                    matrix[2][0], matrix[2][1], matrix[2][2]) / det;

}

void
Matrix::Identity(Matrix& mat) {
  for (int i=0; i<4; i++)
    for (int j=0; j<4; j++)
      mat[i][j] = 0.0f;

  mat[0][0] = mat[1][1] = mat[2][2] = mat[3][3] = 1.0f;
}

void
Matrix::print() const {
  for (int row = 0; row < 4; row++) {
    for (int col = 0; col < 4; col++) {
      printf("%0.4f ", matrix[row][col]);
    }
    printf("\n");
  }
  printf("\n");
}

void
Matrix::WriteToRIB(FILE* output, std::string indent) const {
  fprintf(output, "%sConcatTransform [", indent.c_str());
  for (int row = 0; row < 4; row++) {
    for (int col = 0; col < 4; col++) {
      fprintf(output, "%f ", matrix[row][col]);
    }
    if (row != 3) {
      fprintf(output, "\n%s                 ", indent.c_str());
    }
  }
  fprintf(output, "]\n");
}

void
Matrix::Translate(Matrix& mat, float x, float y, float z) {
  Matrix::Identity(mat);
  mat[3][0] = x;
  mat[3][1] = y;
  mat[3][2] = z;
}

void
Matrix::Scale(Matrix& mat, float x, float y, float z) {
  Matrix::Identity(mat);
  mat[0][0] = x;
  mat[1][1] = y;
  mat[2][2] = z;
}

void
Transform::SetMatrix(Transform& t, const Matrix& m) {
  t.m = m;
  m.Inverse(t.mInv);
}

void
Transform::mult(const Transform& t, Transform& result) const {
  m.multMatrix(t.m, result.m);
  t.mInv.multMatrix(mInv, result.mInv);
}


void
Transform::Identity(Transform& t) {
  Matrix::Identity(t.m);
  Matrix::Identity(t.mInv);
}

void
Transform::Translate(Transform& t, float x, float y, float z) {
  Matrix::Translate(t.m, x, y, z);
  Matrix::Translate(t.mInv, -x, -y, -z);
}

void
Transform::Scale(Transform& t, float x, float y, float z) {
  Matrix::Scale(t.m, x, y, z);
  Matrix::Scale(t.mInv, 1.f / x, 1.f / y, 1.f / z);
}

void
Transform::RotateX(Transform& t, float rad) {
  Matrix::RotateX(t.m, rad);
  Matrix::RotateX(t.mInv, -rad);
}

void
Transform::RotateY(Transform& t, float rad) {
  Matrix::RotateY(t.m, rad);
  Matrix::RotateY(t.mInv, -rad);
}

void
Transform::RotateZ(Transform& t, float rad) {
  Matrix::RotateZ(t.m, rad);
  Matrix::RotateZ(t.mInv, -rad);
}

void
Transform::RotateAxis(Transform& t, const Vec3f& axis, float rad) {
  Matrix::RotateAxis(t.m, axis, rad);
  Matrix::RotateAxis(t.mInv, axis, -rad);
}

void
Matrix::RotateX(Matrix& mat, float rad) {
  Matrix::Identity(mat);
  float cosTheta = cos(rad);
  float sinTheta = sin(rad);
  mat[1][1] = cosTheta;
  mat[1][2] = sinTheta;
  mat[2][1] = -sinTheta;
  mat[2][2] = cosTheta;
}

/*
 * RotateY --
 *
 * Rotation by rad radians about the Y axis.  Rotation is clockwise
 * when looking down the Y axis (looking in the negative Y direction)
 */
void
Matrix::RotateY(Matrix& mat, float rad) {
  Matrix::Identity(mat);
  float cosTheta = cos(rad);
  float sinTheta = sin(rad);
  mat[0][0] = cosTheta;
  mat[0][2] = -sinTheta;
  mat[2][0] = sinTheta;
  mat[2][2] = cosTheta;
}

void
Matrix::RotateZ(Matrix& mat, float theta) {
  Matrix::Identity(mat);
  float cosTheta = cos(theta);
  float sinTheta = sin(theta);
  mat[0][0] = cosTheta;
  mat[0][1] = sinTheta;
  mat[1][0] = -sinTheta;
  mat[1][1] = cosTheta;
}

void
Matrix::RotateAxis(Matrix& mat, const Vec3f& axis, float theta) {
  Matrix::Identity(mat);
  Vec3f axis_norm = axis.normalized();
  float x = axis_norm[0];
  float y = axis_norm[1];
  float z = axis_norm[2];

  float cosTheta = cos(theta);
  float sinTheta = sin(theta);
  float t = 1 - cosTheta;

  mat[0][0] = t * x * x + cosTheta;
  mat[1][0] = t * x * y - sinTheta * z;
  mat[2][0] = t * x * z + sinTheta * z;
  mat[3][0] = 0.f;

  mat[0][1] = t * x * y + sinTheta * z;
  mat[1][1] = t * y * y + cosTheta;
  mat[2][1] = t * y * z - sinTheta * x;
  mat[3][1] = 0.f;

  mat[0][2] = t * x * z - sinTheta * y;
  mat[1][2] = t * y * z + sinTheta * x;
  mat[2][2] = t * z * z + cosTheta;
  mat[3][2] = 0.f;

  mat[0][3] = 0.f;
  mat[1][3] = 0.f;
  mat[2][3] = 0.f;
  mat[3][3] = 1.f;
}

/*
 * Lookat --
 *
 * Produces a lookat matrix (world to camera space).  Post-multiplying
 * a world space row vector by the resulting matrix yields a camera
 * space row vector.
 */
void
Matrix::Lookat(Matrix& mat,
               const Vec3f& eye, const Vec3f& at, const Vec3f& up) {

  Vec3f xaxis, yaxis, zaxis;

  // The camera coordinate system is defined by:
  // x-axis is off to the right (perpendicular to the view direction and up vector)
  // y-axis is up
  // z-axis is coming at the camera (camera looks in negative Z direction)

  zaxis = eye - at;
  zaxis.normalize();
  xaxis = up.cross(zaxis);
  xaxis.normalize();
  yaxis = zaxis.cross(xaxis);
  yaxis.normalize();

  mat[0][0] = xaxis[0];
  mat[1][0] = xaxis[1];
  mat[2][0] = xaxis[2];
  mat[3][0] = -1.f * xaxis.dot(eye);

  mat[0][1] = yaxis[0];
  mat[1][1] = yaxis[1];
  mat[2][1] = yaxis[2];
  mat[3][1] = -1.f * yaxis.dot(eye);

  mat[0][2] = zaxis[0];
  mat[1][2] = zaxis[1];
  mat[2][2] = zaxis[2];
  mat[3][2] = -1.f * zaxis.dot(eye);

  mat[0][3] = 0.f;
  mat[1][3] = 0.f;
  mat[2][3] = 0.f;
  mat[3][3] = 1.f;
}

/*
 * LookatInv --
 *
 * Generate inverse of a lookat matrix (generates a camera to world
 * transformation matrix)
 */
void
Matrix::LookatInv(Matrix& mat, const Vec3f& eye, const Vec3f& at, const Vec3f& up) {

  // NOTE(boulos): Checked that both these versions produce the same
  // answer.
#if 1
  Vec3f xaxis, yaxis, zaxis;

  zaxis = eye - at;
  zaxis.normalize();
  xaxis = up.cross(zaxis);
  xaxis.normalize();
  yaxis = zaxis.cross(xaxis);
  yaxis.normalize();

  Matrix::Identity(mat);

  for (int i=0; i<3; i++) {
    mat[0][i] = xaxis[i];
    mat[1][i] = yaxis[i];
    mat[2][i] = zaxis[i];
  }

  mat[3][0] = eye[0];
  mat[3][1] = eye[1];
  mat[3][2] = eye[2];
#else
  Matrix temp;
  Matrix::Lookat(temp, eye, at, up);
  temp.Inverse(mat);
#endif
}

void
BBox3f::Transform(const Matrix& m, BBox3f& out) const {

  Vec3f p;

  out.Reset();
  m.TransformPoint(bmin, p);
  out.AddPoint(p);
  m.TransformPoint( Vec3f(bmax[0], bmin[1], bmin[2]), p);
  out.AddPoint(p);
  m.TransformPoint( Vec3f(bmax[0], bmax[1], bmin[2]), p);
  out.AddPoint(p);
  m.TransformPoint( Vec3f(bmin[0], bmax[1], bmin[2]), p);
  out.AddPoint(p);

  m.TransformPoint(bmax, p);
  out.AddPoint(p);
  m.TransformPoint( Vec3f(bmin[0], bmin[1], bmax[2]), p);
  out.AddPoint(p);
  m.TransformPoint( Vec3f(bmax[0], bmin[1], bmax[2]), p);
  out.AddPoint(p);
  m.TransformPoint( Vec3f(bmin[0], bmax[1], bmax[2]), p);
  out.AddPoint(p);
}
