#include "flow.h"

#include <algorithm>
#include <limits>
#include <queue>
#include <stack>

#include <assert.h>

FlowSolver::FlowSolver(const Graph& g, int source, int sink)
  : _source(source), _sink(sink)
{
  const std::vector<Vertex>& v = g.v;
  const std::vector<Edge>& e = g.e;

  _resV.resize(v.size());
  /*
  for (size_t i = 0; i < v.size(); i++) {
    _resV[i] = ...
  }
  */

  _resE.resize(e.size());
  for (size_t i = 0; i < e.size(); i++) {
    int start = e[i].start;
    int end = e[i].end;
    std::pair<int, int> edgePair(start, end);

    _resE[i].start = start;
    _resE[i].end = end;
    _resE[i].capacity = e[i].capacity;
    _resE[i].orig_capacity = e[i].capacity;

    // hideous
    std::pair<std::map<std::pair<int, int>, int>::iterator, bool> v = _toEdge.insert(std::make_pair(edgePair, i));

    if (!v.second) { // already existed
      throw "badness.  duplicate edge in graph";
    }
  }

  _newEdgeBegin = _resE.size(); // where new edges start

  int nextEdgeIndex = (int)_newEdgeBegin;
  // now create back edges
  for (size_t i = 0; i < e.size(); i++) {
    int start = e[i].start;
    int end = e[i].end;
    std::pair<int, int> backEdgePair(end, start);

    if (_toEdge.find(backEdgePair) == _toEdge.end()) { // doesn't exist
      ResEdge backEdge;
      backEdge.start = end;
      backEdge.end = start;
      backEdge.capacity = 0;
      _resE.push_back(backEdge);

      _toEdge[backEdgePair] = nextEdgeIndex++;
    }
  }

  assert (nextEdgeIndex == (int)_resE.size());

  // connect verts to edges
  for (size_t i = 0; i < e.size(); i++) {
    int start = e[i].start;
    int end = e[i].end;
    std::pair<int, int> edgePair(start, end);

    assert (_toEdge.find(edgePair) != _toEdge.end());

    int edgeIndex = _toEdge[edgePair];

    _resV[start].outEdgeIndices.push_back(edgeIndex);
  }
}

void
FlowSolver::DebugPrintState() const
{
  fprintf(stderr, "edge state\n");
  for (size_t i = 0; i < _resE.size(); i++) {
    fprintf(stderr, "%zu (%d %d): %d\n",
        i,
      _resE[i].start,
      _resE[i].end,
      _resE[i].capacity);
  }
}

std::vector<int> 
FlowSolver::MinCut()
{
  while (true) {

    EdgePath path = FindAugmentingPath();

    int capacity = GetPathCapacity(path);
    if (capacity == 0) {
      break;
    }

    UpdateFlow(path, capacity);

  }

  std::vector<int> ret;
  for (size_t i = 0; i < _newEdgeBegin; i++) {
    if (_resE[i].capacity == 0) {
      ret.push_back(i);
    }
  }
  return ret;
}

struct DijkstraNode {
  int capacity;
  int vertexIndex;
  int prevIndex;

  bool operator<(const DijkstraNode& other) const { 
    return capacity < other.capacity;
  }
};

FlowSolver::EdgePath
FlowSolver::FindAugmentingPath() const
{
  // dijkstra for finding the best augmenting path

  // Initialize dijkstra
  std::vector<int> bestCapacities(_resV.size(), std::numeric_limits<int>::min());
  std::vector<int> previous(_resV.size(), -1);

  std::priority_queue<DijkstraNode> pq;
  DijkstraNode startNode;
  startNode.capacity = 0;
  startNode.vertexIndex = _source;
  startNode.prevIndex = -1;
  pq.push(startNode);

  while (!pq.empty()) {
    DijkstraNode n = pq.top();
    pq.pop();

    int vertexIndex = n.vertexIndex;
    if (n.capacity < bestCapacities[vertexIndex]) continue;
    bestCapacities[vertexIndex] = n.capacity;
    previous[vertexIndex] = n.prevIndex;

    if (vertexIndex == _sink) break; // done!

    const ResVertex& v = _resV[vertexIndex];
    for (size_t i = 0; i < v.outEdgeIndices.size(); i++) {
      const ResEdge& e = _resE[v.outEdgeIndices[i]];
      if (e.capacity <= 0) continue; // early out.  technically don't need this

      DijkstraNode newNode;
      newNode.vertexIndex = e.end;
      newNode.capacity = std::min(e.capacity, n.capacity);
      newNode.prevIndex = vertexIndex;
      pq.push(newNode);
    }
  }

  // get path along verticies from sink to source
  std::vector<int> vertPath;
  int curr = _sink;
  while (true) {
    vertPath.push_back(curr);
    if (previous[curr] < 0) break;
    curr = previous[curr];
  }

  EdgePath ret;
  if (curr != _source) return ret; // didn't find a way home

  std::reverse(vertPath.begin(), vertPath.end());
  int start = _source;
  for (size_t i = 1; i < vertPath.size(); i++) {
    int end = vertPath[i];

    std::map<std::pair<int, int>, int>::const_iterator iter = _toEdge.find(std::make_pair(start, end));

    assert (iter != _toEdge.end());

    int edgeIndex = iter->second;
    ret.indices.push_back(edgeIndex);

    start = end;
  }
  return ret;
}

int
FlowSolver::GetPathCapacity(const EdgePath& path) const
{
  const std::vector<int>& edgeIndices = path.indices;

  if (edgeIndices.empty()) return 0;

  int minCapacity = _resE[edgeIndices[0]].capacity;

  //std::vector<int>::const_iterator min_elem = std::min_element(edgeIndices.begin(), edgeIndices.end());
  for (size_t i = 0; i < edgeIndices.size(); i++) {
    minCapacity = std::min(minCapacity, _resE[edgeIndices[i]].capacity);
  }

  return minCapacity;
}

void
FlowSolver::UpdateFlow(const EdgePath& path, int capacity)
{
  const std::vector<int>& edgeIndices = path.indices;
  for (size_t i = 0; i < edgeIndices.size(); i++) {
    ResEdge& forwardEdge = _resE[edgeIndices[i]];

    int start = forwardEdge.start;
    int end = forwardEdge.end;
    int backEdgeIndex = _toEdge[std::make_pair(end, start)];
    ResEdge& backEdge = _resE[backEdgeIndex];

    forwardEdge.capacity -= capacity;
    backEdge.capacity += capacity;
  }
}

