#ifndef __BLUR_H__
#define __BLUR_H__

#include <core/geom.h>
#include <core/sifter.h>
#include <core/matcher.h>
#include <gl/platform.h>

#include <cv.h>

#include <algorithm>
#include <math.h>
#include <vector>

class MyViewer {
  public:
    MyViewer(int windowId, 
        int windowWidth, int windowHeight);
    void SetWindowSize(int windowWidth, int windowHeight);
    void Draw() const;

    void KeyPress(unsigned char key, bool down, int x, int y);
    void SpecialKeyPress(int key, int x, int y);
    void MouseButton(int button, int state, int x, int y);

    // returns true if redraw needed
    bool MouseMotion(int x, int y);

    void AddImage(const Sifter::Image& img);
    void SetOriginal(const cv::Mat& original) {
      _original = original;
      _final = original;
    }

    void SetCurrent(int index) {
      _currImg = index;

      if (_currImg < 0) _currImg = _images.size() - 1;
      if (_currImg >= (int)_images.size()) _currImg = 0;
    }

  private:
    int _windowId;
    int _windowSize[2];

    // window pan position
    bool _canPan;
    Vec2f _windowPan;

    float _zoom;
    float _lastZoom;
    float _lastZoomR;

    Vec2f _selectStart;

    // size of the "canvas".  This sits inside the window.
    int _canvasSize[2];

    typedef enum {
      None = 0,
      Select,
      Pan,
      Zoom, 
      Matte
    } MouseAction;
    MouseAction _action;

    Vec2f _lastPos;

    int _currImg;

    std::set<int> _selFeatureIds;

    struct ImageRecord {
      ImageRecord() :
        orig_tex(0), xform_tex(0)
      {
      }

      // these should be const
      Sifter::Image orig_img;
      GLint orig_tex;

      // these are computed "per-match" operation
      Matcher::Matches matches;
      cv::Mat xform;
      Sifter::Image xform_img;
      GLint xform_tex;
    };
    std::vector<ImageRecord> _images;

    float _matteThreshhold;
    cv::Mat _matte;
    cv::Mat _blurredImage;
    cv::Mat _original;
    cv::Mat _final;

    bool _showXformed;
    bool _aligned;
    bool _showImage;
    bool _showFeatures;
    bool _showUnmatched;

    typedef enum {
      Single = 0,
      SideBySide,
      Overlay,
      Final,
      Original,
      Blurred,
      Alpha
    } ViewMode;

    ViewMode _viewMode;

    // NOTE(edluong): Since we use this for zooming, we use _lastZoom which
    // should always be in sync with _zoom, unless we are zooming.    
    inline float CenterDistance(float x, float y) const {
      // canvas center in window coords.  remember, y is 0 at the top
      //float cx = _windowSize[0]/2.f + (_canvasSize[0]/2.f - _windowPan[0]) * _lastZoom;
      //float cy = _windowSize[1]/2.f - (_canvasSize[1]/2.f - _windowPan[1]) * _lastZoom;

      float cx = _windowSize[0]/2.f;
      float cy = _windowSize[1]/2.f;
      float rx = x - cx;
      float ry = y - cy;

      return std::max((float)sqrt(rx*rx + ry*ry), 1.f);
    }

    inline Vec2f screenToWorld(const Vec2f& p) const {
      Vec2f windowCenter(_windowSize[0]/2.f, _windowSize[1]/2.f);
      Vec2f diff = (((p - windowCenter)) / _zoom);
      diff[1] *= -1.f;

      return _windowPan + diff;
    }

    inline Vec2f worldToImage(const Vec2f& p, int which) const {
      // TODO(edluong): this also depends on the mode
      Vec2f ret = p;
      ret[1] *= -1.f;

      return ret + Vec2f(0.f, _images[which].orig_img.m.size().height);
    }

    void DrawImagePixels(const cv::Mat& imgData, float alpha = 1.f) const; // draws the 0th image to the side
    void DrawImage(int which, bool showXformed, float alpha = 1.f, bool drawText = true) const;
    void DrawSideImage(bool otherXformed) const; // draws the 0th image to the side
    void DrawFeatures(int which, bool showXformed) const;
    void DrawMatches(bool otherXformed) const;
    void DrawMatches(bool otherXformed, bool skipUnmatched) const;
    void DrawSelectBox() const;
    void DrawHeatmap(float width, float height, int numIntervals = kHeatmapIntervals) const;

    void PickFeatures(int which, const BBox2f& box);

    // returns selection box in world space
    BBox2f GetSelectBox() const;

    void AlignImages(bool useRansac);
    void BlendImages();
    void ClearBlur();

    void InitByteToFloatLUT() {
      for (int b = 0; b < 256; b++) {
        byteToFloatLUT[b] = static_cast<float>(b)/255.f;
      }
    }
    float ByteToFloat(unsigned char b) const { return byteToFloatLUT[b]; }
    float byteToFloatLUT[256];

    cv::Mat ComputeChangeMask(const cv::Mat& ref, const cv::Mat& changed, float threshhold);
    void MakeFinal(cv::Mat& final, const cv::Mat& orig, const cv::Mat& blurred, const cv::Mat& matte);

    static const int kHeatmapIntervals = 10;
};

#endif // __BLUR_H__

